//------------------------------------------------------------------------------
// File: OutpostWood.cs
// Handles wood trade outpost mission
// Author: Michael Felice
//------------------------------------------------------------------------------


// SCENARIO CALLBACK FUNCTIONS for Outpost Wood Mission

// load scenario function (initializes the outpost wood mission tasks)
function startOutpostWoodTasks()
{
   tsStartTask("OutpostWood", "OutpostWoodInit");
   
   // disable the train station upgrade
   tsEnableCommand(TrainStationUpgradeCmdData, false);
   
   // disable trains from moving on this map
   slgTrain_DisableTrains();
   
   // disable the quit scenario button
   $DisableQuitScenario = true;
}

// quit scenario function (turns off the outpost wood task)
function quitOutpostWoodTasks()
{
   tsEndTask("OutpostWood");
   
   // disable wood production
   tsEnableCommand(ProduceWoodCmdData, false);
   
   // check if the building should be unselected
   if (csIsCommandDisabled(ProduceGoldCmdData) == true &&
      csIsCommandDisabled(ProduceFoodCmdData) == true)
   {
      %object = nameToId("mapplatform");
      %building = slgQueryInterface(%object, $CID_BUILDING);
      %building.startExclusiveProduce($Resource::Count);
   }
   
   // reset the wood trigger and resend the alert message
   TaskModifier.strResetVertex("OutpostTasks", "OutpostTaskWoodTrigger");
   TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskWoodTrigger", $TSV_AVAIL);
   
   // resend the alert message
   SendOutpostTaskAlert("ID_TASKNAME_OUTPOSTWOOD", "id_outpost_wood");
}


// MAIN SET OF TASKS

// explain the goal of the map, set up quests
function OutpostWoodInit::onExit(%vertex)
{
   // send the information
   tsSendTaskPopUpDlg("ID_TASKNAME_OUTPOSTWOOD", "id_outpostwood_intro",
      $TaskButton::Okay, HeroCharacter, "id_outpostwood_intro");
   
   // set up the quest
   tsAddQuestToTracker(true, "Outpost Wood Mission",
      "Talk to " @ HeroCharacter.name @ " to find out what we need to do before we can import Wood at our town's Outpost.",
      "Before you can start receiving shipments of Wood back at town, you will need to establish a source for your town's additional Wood supply.",
      "Reward: Import Wood at your Outpost and 10 Experience");
   tsReadyQuest("Outpost Wood Mission");
}

// tell the player how to start the quest
function OutpostWoodStart::onExit(%vertex)
{
   // update the base quest information
   tsAddTaskToTracker("Outpost Wood Mission", "Establish a source of additional Wood.",
      "Complete all of the necessary tasks to establish a source of additional Wood for your town.");
   
   // add the new tasks for building the woodhuts and the lumber mill
   tsAddTaskToTracker("Outpost Wood Mission", "Build three Wood Huts.",
      "Build three Wood Huts and a Lumber Mill to set up a source of Wood.");
   tsAddTaskToTracker("Outpost Wood Mission", "Build a Lumber Mill.",
      "Build three Wood Huts and a Lumber Mill to set up a source of Wood.");
}



// HANDLES BUILDING AND DESTROYING THREE WOODHUTS

// this occurs when one woodhut has been created
function OutpostWoodWHCreate1::onExit(%vertex)
{
}

// this occurs when two woodhuts have been created
function OutpostWoodWHCreate2::onExit(%vertex)
{
}

// this occurs when three woodhuts have been created
function OutpostWoodWHCreate3::onExit(%vertex)
{
   tsCompleteTaskInTracker("Outpost Wood Mission", "Build three Wood Huts.");
   
   // check if the lumber mill is also completed
   if (OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodOPStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPStart1", $TSV_CMPLETE);
   }
   
   // check if the woodhuts and lumber mill are also completed
   if (OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodOPCreate.isMarkedComplete() == true &&
      OutpostWoodResourceStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceStart1", $TSV_CMPLETE);
   }
   
   // check if the woodhuts, lumber mill, and outpost are created
   // and if so, turn off all vertices for destruction and turn on the
   // completion vertices
   if (OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodOPCreate.isMarkedComplete() == true &&
      OutpostWoodResourceCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodComplete1", $TSV_AVAIL);
   }
}

// this occurs when a woodhut is destroyed
function OutpostWoodWHDestroy::onMessage(%vertex, %type, %param)
{
   // check if a wood hut is destroyed, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was not affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHDestroy", $TSV_AVAIL);
   
   // find out how many active woodhuts are remaining
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   %woodhutCount = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // get the object in the list
      %object = %list.getId(%index);
      
      // if the object datablock is not a wood hut, move to the next object
      %datablock = %object.getDataBlock();
      if (%datablock.getName() !$= "Woodhut")
      {
         continue;
      }
      
      // if the woodhut is in the productino state, add to the woodhut count
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %woodhutCount++;
      }
   }
   
   // update the vertices based on how many woodhuts are available
   switch (%woodhutCount)
   {
      case 0:
         // no woodhuts are active
         TaskModifier.strResetVertex("OutpostWood", "OutpostWoodWHCreate1");
         TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHCreate1", $TSV_AVAIL);

      case 1:
         // one woodhut is active
         TaskModifier.strResetVertex("OutpostWood", "OutpostWoodWHCreate2");
         TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHCreate2", $TSV_AVAIL);

      case 2:
         // two woodhuts are active (lose the woodhut completion message)
         TaskModifier.strResetVertex("OutpostWood", "OutpostWoodWHCreate3");
         TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHCreate3", $TSV_AVAIL);
         tsAddTaskToTracker("Outpost Wood Mission", "Build three Wood Huts.",
            "Build three Wood Huts and a Lumber Mill to set up a source of Wood.");
   }
}



// HANDLES BUILDING AND DESTROYING THE LUMBER MILL

// this occurs when the first lumber mill has been created
function OutpostWoodLMCreate::onExit(%vertex)
{
   // complete the lumber mill creation task
   tsCompleteTaskInTracker("Outpost Wood Mission", "Build a Lumber Mill.");
   
   // check if the woodhuts are also completed
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodOPStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPStart1", $TSV_CMPLETE);
   }
   
   // check if the woodhuts and lumber mill are also completed
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodOPCreate.isMarkedComplete() == true &&
      OutpostWoodResourceStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceStart1", $TSV_CMPLETE);
   }
   
   // check if the woodhuts, lumber mill, and outpost are created
   // and if so, turn off all vertices for destruction and turn on the
   // completion vertices
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodOPCreate.isMarkedComplete() == true &&
      OutpostWoodResourceCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodComplete1", $TSV_AVAIL);
   }
}

// this occurs when a woodhut is destroyed
function OutpostWoodLMDestroy::onMessage(%vertex, %type, %param)
{
   // check if a lumber mill is destroyed, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was no affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMDestroy", $TSV_AVAIL);
   
   // find out how many active woodhuts are remaining
   %list = slgGetServerBuildingList();
   %count = %list.getSize();
   %lumbermillCount = 0;
   for (%index = 0; %index < %count; %index++)
   {
      // get the object in the list
      %object = %list.getId(%index);
      
      // if the object datablock is not a wood hut, move to the next object
      %datablock = %object.getDataBlock();
      if (%datablock.getName() !$= "Lumbermill")
      {
         continue;
      }
      
      // if the woodhut is in the productino state, add to the woodhut count
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %lumbermillCount++;
      }
   }
   
   // update the vertices based on how many woodhuts are available
   switch (%lumbermillCount)
   {
      case 0:
         // no lumber mills are active (lose the lumber mill completion message)
         TaskModifier.strResetVertex("OutpostWood", "OutpostWoodLMCreate");
         TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMCreate", $TSV_AVAIL);
         tsAddTaskToTracker("Outpost Wood Mission", "Build a Lumber Mill.",
            "Build three Wood Huts and a Lumber Mill to set up a source of Wood.");
   }
}





// when the previous conditions are met for the first time, and the
// platform needs to be upgraded
function OutpostWoodOPStart1::onExit(%vertex)
{
   // give the hero something to talk about
   tsSendTaskClickDlg("ID_TASKNAME_OUTPOSTWOOD", "id_outpostwood_platform",
      $TaskButton::Okay, HeroCharacter, "id_outpostwood_platform");
}

// set up the platform for the upgrade
function OutpostWoodOPStart2::onExit(%vertex)
{
   // update the quest information
   tsAddTaskToTracker("Outpost Wood Mission", "Upgrade the Platform to an Outpost.",
      "Upgrade the Platform to an Outpost in order to ship Wood back to town.");
}

// before the outpost can be created, the platform team needs to be set
function OutpostWoodOPCreate::onEnter(%vertex)
{
   // update the team of the platform
   resourcePlatform.setTeam($OST_PLAYER);
}

// this occurs when the outpost has been upgraded
function OutpostWoodOPCreate::onExit(%vertex)
{
   tsCompleteTaskInTracker("Outpost Wood Mission", "Upgrade the Platform to an Outpost.");
   
   // check if the woodhuts and lumber mill are also completed
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodResourceStart1.isMarkedComplete() == false)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceStart1", $TSV_CMPLETE);
   }
   
   // check if the woodhuts, lumber mill, and outpost are created
   // and if so, turn off all vertices for destruction and turn on the
   // completion vertices
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodResourceCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodComplete1", $TSV_AVAIL);
   }
}

// this occurs when the outpost is destroyed
function OutpostWoodOPDestroy::onMessage(%vertex, %type, %param)
{
   // check if the outpost is destroyed, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was no affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPDestroy", $TSV_AVAIL);
   
   // if the outpost is lost, we know it is the player's outpost because
   // it is the only outpost on the map, so reactivate the outpost task
   TaskModifier.strResetVertex("OutpostWood", "OutpostWoodOPCreate");
   TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPCreate", $TSV_AVAIL);
   tsAddTaskToTracker("Outpost Wood Mission", "Upgrade the Platform to an Outpost.",
      "Upgrade the Platform to an Outpost in order to ship Wood back to town.");
}



// HANDLES RESOURCE COLLECTION

// after the lumber mill, woodhuts, and outpost have been created, send
// a message for resource collection
function OutpostWoodResourceStart1::onExit(%vertex)
{
   // give the hero something to talk about
   tsSendTaskClickDlg("ID_TASKNAME_OUTPOSTWOOD", "id_outpostwood_resource",
      $TaskButton::Okay, HeroCharacter, "id_outpostwood_resource");
}

// set up messages for resource collection
function OutpostWoodResourceStart2::onExit(%vertex)
{
   // update the quest information
   tsAddTaskToTracker("Outpost Wood Mission", "Collect 400 Wood.",
      "A stockpile of 400 Wood should be enough to begin sending Wood back to town.");
}

// check if wood resources have passed 400
function OutpostWoodResourceCreate::onUpdate(%vertex)
{
   // if not enough wood has been collected, do nothing
   if (rsGetResourceCount($Resource::Wood) < 400)
   {
      return;
   }
   
   // complete this step of resource collection
   TaskModifier.StrMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_CMPLETE);
   TaskModifier.StrMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_AVAIL);
   
   // update the task to completion
   tsCompleteTaskInTracker("Outpost Wood Mission", "Collect 400 Wood.",
      "A stockpile of 400 Wood should be enough to begin sending Wood back to town.");
   
   // check if the woodhuts, lumber mill, and outpost are created
   // and if so, turn off all vertices for destruction and turn on the
   // completion vertices
   if (OutpostWoodWHCreate3.isMarkedComplete() == true &&
      OutpostWoodLMCreate.isMarkedComplete() == true &&
      OutpostWoodOPCreate.isMarkedComplete() == true)
   {
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodWHDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodLMDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodOPDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_CLOSED);
      TaskModifier.strMarkVertex("OutpostWood", "OutpostWoodComplete1", $TSV_AVAIL);
   }
}

// check if wood resources have falled below 400
function OutpostWoodResourceDestroy::onUpdate(%vertex)
{
   // if enough wood has been collected, do nothing
   if (rsGetResourceCount($Resource::Wood) >= 400)
   {
      return;
   }
   
   // reactivate this step of resource collection
   TaskModifier.StrMarkVertex("OutpostWood", "OutpostWoodResourceCreate", $TSV_AVAIL);
   TaskModifier.StrMarkVertex("OutpostWood", "OutpostWoodResourceDestroy", $TSV_CMPLETE);
   
   // update the task to incompletion
   tsAddTaskToTracker("Outpost Wood Mission", "Collect 400 Wood.",
      "400 planks of Wood should be enough to begin sending Wood back to town.");
}



// HANDLES OUTPOST WOOD COMPLETION

// notify the player that mission is done
function OutpostWoodComplete1::onEnter(%vertex)
{
   // give the hero something to talk about
   tsSendTaskPopUpDlg("ID_TASKNAME_OUTPOSTWOOD", "id_outpostwood_complete",
      $TaskButton::Okay, HeroCharacter, "id_outpostwood_complete");
}

// close the quests that have been completed and update experience
function OutpostWoodComplete1::onExit(%vertex)
{
   // complete the quest
   tsCompleteTaskInTracker("Outpost Wood Mission", "Establish a source of additional Wood.");
   
   // experience reward for completing the quest
   tsModifyResource($Resource::Experience, 10, true);
   SendProductionToClient(HeroCharacter, "10 experience");
}

// exit the satellite
function OutpostWoodComplete2::onExit(%vertex)
{
   // return from this satellite
   %scenario = new SLScenario()
   {
        stack = false;
   };
   LoadScenario(%scenario);
   
   // update the satellite flag
   mapsatellite_outpostwood.disabled = false;
   TaskModifier.strMarkVertex("Badgetasks", "OutpostTaskWoodComplete", $TSV_CMPLETE);   TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskWoodTrigger", $TSV_CMPLETE);
}
